/*
 About: Author
 Tomasz Scislo
 About: Date
 2010-10-06
 */
/*
 * Class: CommonFileSystem
 *
 * Class which implements common area file system support
 *
 *  Properties:
 *  
 *   fileSystemObj - Samsung Framework FileSystemObject, used for common file area operations
 *   
 *   widgetPath - common file area directory name, where XML local file will be stored
 *   
 *   widgetFile - name of the XML settings file
 *   
 *   fileObjRead - Samsung Framework file object for read purpouse
 *   
 *   fileObjWrite - Samsung Framework file object for write purpouse
 *   
 *   xmlObj - (xmlObject) XML settings file object representation
 *   
 *   xmlDamaged - (boolean) flag indicating if XML file was damaged during writing and if so the default one is resotred 
 *
 */
var CommonFileSystem = function(){

    this.fileSystemObj = new FileSystem();
	this.widgetPath = "NG_USA_WIDGET_animal"; 
    this.widgetFile = "NG_SETTINGS.xml";
    this.fileObjRead;
    this.fileObjWrite;
    this.xmlObj;
    this.xmlDamaged = false;
    
    /*
     * Function: log
     *
     * Simple logger
     *
     * Parameters:
     * msg - string to log
     *
     *
     * Returns:
     */
    this.log = function(msg){
        alert(">>>>>>>> CommonFileSystem: " + msg);
    }
    this.log("CommonFileSystem");
    /*
     * Function: pathExists
     *
     * Checks if directory path exists in common area.
     *
     * Parameters:
     * path - (string) path to verify
     *
     *
     * Returns:
     * boolean - true if path exists common area exists, false otherwise
     */
    this.pathExists = function(path){
        return this.fileSystemObj.isValidCommonPath(path);
    };
    
    /*
     * Function: createDir
     *
     * Creates directory in common area.
     *
     * Parameters:
     * path - (string) path to verify
     *
     *
     * Returns:
     * boolean - true if new directory creation was possitive, false otherwise
     */
    this.createDir = function(dir){
        return this.fileSystemObj.createCommonDir(dir);
    };
    
    /*
     * Function: writeFile
     *
     * Replaces all data in widgetFile with new string
     *
     * Parameters:
     * serializedXML - string to write into the file
     *
     *
     * Returns:
     */
    this.writeFile = function(serializedXML){
        this.log('writeFile');
        if (!(this.fileObjWrite = this.fileSystemObj.openCommonFile(this.widgetPath + "/" + this.widgetFile, 'w+'))) 
            throw new Error('Unable to open XML file for write in common directory! Unable to proceed!');
        if (!this.fileObjWrite.writeAll(serializedXML)) 
            throw new Error('Unable to write data in XML file in common directory! Unable to proceed!');
        this.fileSystemObj.closeCommonFile(this.fileObjWrite);
    };
    
    /*
     * Function: readAll
     *
     * Reads the content of the widgetFile
     *
     * Parameters:
     *
     *
     * Returns:
     * string - content of file
     */
    this.readFile = function(){
        var retStr = "";
        var strLine;
        while ((strLine = this.fileObjRead.readLine())) {
            retStr += strLine;
        }
        return retStr;
    };
    
    /*
     * Function: createDefaultXML
     *
     * Creates default XML structure for first widget execution
     *
     * Parameters:
     *
     * Returns:
     * xmlObject
     *
     */
    this.createDefaultXML = function(){
		this.log("createDefaultXML");
        var parser = new DOMParser();
        
        var xml = '<?xml version="1.0" encoding="utf-8"?>' +
        '<nationalgeographic>' +
        '<effectid>0</effectid>' +
        '<timeoutid>0</timeoutid>' +
        '</nationalgeographic>';
        
        return parser.parseFromString(xml, "text/xml");
    };
    
    /*
     * Function: serializeXML
     *
     * Simple serialization method which handels only NG XML structure
     *
     * Parameters:
     * xmlObject - xmlObject to serialize
     *
     * Returns:
     * string - serialized xmlObject
     */
    this.serializeXML = function(xmlObject){
        var serialized = "";
        serialized = '<?xml version="1.0" encoding="utf-8"?> \n' +
        '<nationalgeographic> \n';
        
        if ($(xmlObject).find('nationalgeographic').children().length == 0) 
            throw new Error('No nodes in xmlObject to serialize!');
        
        $(xmlObject).find('nationalgeographic').children().each(function(i){
            serialized += "<" + this.nodeName + ">" + $(this).text() + "</" + this.nodeName + "> \n";
        });
        
        serialized += '</nationalgeographic>';
        
        return serialized;
    }
    
    /*
     * Function: parseSettingsXML
     *
     * Method to parse XML in search for item nodes, which indicates pictureid's of images that where set inactive in widget
     *
     * Parameters:
     * xmlObj - xmlObject to parse
     *
     * Returns:
     * boolean (false) when no node item were found, JSON if ok
     */
    this.parseSettingsXML = function(xmlObj){
        var arr = Array();
        if ($(xmlObj).find('nationalgeographic').children('item').length == 0) {
            this.log("No inactive images found in XML!");
            arr = undefined;
        }
        
        $(xmlObj).find('nationalgeographic').children('item').each(function(i){
            arr[i] = $.trim($(this).text());
        });
        
        if ($(xmlObj).find('nationalgeographic').children('effectid').length == 0) {
            this.xmlDamaged = true;
            this.log("No effectid in XML, using default 0 value!");
            var effectid = 0;
        }
        else {
            var effectid = parseInt($.trim($(xmlObj).find('nationalgeographic').children('effectid').text()));
        }
        if ($(xmlObj).find('nationalgeographic').children('timeoutid').length == 0) {
            this.xmlDamaged = true;
            this.log("No timeoutid in XML, using default 0 value!");
            var timeoutid = 0;
        }
        else 
            var timeoutid = parseInt($.trim($(xmlObj).find('nationalgeographic').children('timeoutid').text()));
        
        return {
            items: arr,
            effectid: effectid,
            timeoutid: timeoutid
        };
    }
    
    /*
     * Function: upDateNGImageViewerAndGlobals
     *
     * Method to update NGImageViewer JSON array and global init variables
     *
     * Parameters:
     * parsedJson - JSON parsed from XML
     *
     * Returns:
     */
    this.upDateNGImageViewerAndGlobals = function(parsedJson){
        activeTranistion = parsedJson.effectid;
        speed = parsedJson.timeoutid;
        this.log("activeTranistion form XML: " + activeTranistion);
        this.log("speed form XML: " + speed);
        if (typeof parsedJson.items !== "undefined") {
            for (var i = 0; i < parsedJson.items.length; i++) {
                for (var k = 0; k < NGImageViewer.images.length; k++) {
                    if (parsedJson.items[i] === NGImageViewer.images[k].name) {
                        this.log("Inactive image found in XML: " + NGImageViewer.images[k].name);
                        NGImageViewer.images[k].active = false;
                    }
                }
            }
        }
    }
    
    /*
     * Function: writeLocalXML
     *
     * Method to write local XML settings file
     *
     * Parameters:
     *
     * Returns:
     */
    this.writeLocalXML = function(){
        this.log("Writing current widget state into local XML file!");
		this.xmlObj=this.createDefaultXML();
        $(this.xmlObj).find('nationalgeographic').children('effectid').text(activeTranistion.toString());
        $(this.xmlObj).find('nationalgeographic').children('timeoutid').text(speed.toString());
        $(this.xmlObj).find('nationalgeographic').children('item').remove();
        for (var i = 0; i < NGImageViewer.images.length; i++) {
            if (NGImageViewer.images[i].active == false) {
                var node = document.createElement('item');
                node.appendChild(document.createTextNode(NGImageViewer.images[i].name));
                if (!this.xmlObj.getElementsByTagName('nationalgeographic')[0].appendChild(node)) 
                    throw new Error("Inactive pictures found, but no XML nodes created for them!");
            }
        }
        
        this.writeFile(this.serializeXML(this.xmlObj));
    }
    
    /*
     * Function: processLocalXML
     *
     * Main method used to process Local XML file
     *
     * Parameters:
     *
     * Returns:
     */
    this.processLocalXML = function(){
        //if widgetPath doesn't exist we assume that this is first widget execution
        if (!this.pathExists(this.widgetPath)) {
            this.log("First widget execution!");
            if (!this.createDir(this.widgetPath)) 
                throw new Error("Unable to create widget common directory! Unable to proceed!");
            var defXML = this.createDefaultXML();
            this.writeFile(this.serializeXML(defXML));
            this.xmlObj = defXML;
            
        }
        //reading widget settings from XML file
        else {
            this.log("Reading Widget XML");
            if (!(this.fileObjRead = this.fileSystemObj.openCommonFile(this.widgetPath + "/" + this.widgetFile, 'r+'))) 
                throw new Error('Unable to open XML file for read in common directory! Unable to proceed!');
            var xmlString = this.readFile();
            var parser = new DOMParser();
            this.xmlObj = parser.parseFromString(xmlString, "text/xml");
            this.upDateNGImageViewerAndGlobals(this.parseSettingsXML(this.xmlObj));
			
			// if widget damages the XML, new Default one is created
            if (this.xmlDamaged) {
				this.log("Damaged XML!");
				this.xmlObj=this.createDefaultXML();
				this.writeFile(this.serializeXML(this.xmlObj));
				this.xmlDamaged=false;
			}
        }
    }
    
}
